#!/usr/bin/python
#*******************************************************************************
#  synertronixx GmbH
#  Lange Laube Str. 22
#  30159 Hannover
#  Tel. : 0511 / 262 999  0
#  Fax. : 0511 / 262 999 29
#  Web  : www.synertronixx.de
#  eMail: devilanapp@synertronixx.de
# ----------------------------
# Project       : Raspberry Pi Finder to use with iOS App DeviLAN 
# Name          : devilan_pi_finder.py
# IDLE          : version 2.7.3 , Python version 2.7.3 & Tk version 8.5                   
# Version       : 1.1
# Date          : 24.04.2014
# Author        : DeviLAN app team
# Description   :
#
# - This program allows to find Raspberry Pi modules in your LAN via an
#   UDP broadcast identification mechanism.
#
# - Use iOS App DeviLAN (TCP/IP based Remote Control tool) to find and control
#   Raspberry Pis which you can download for free from the Appstore
#   http://itunes.apple.com/us/app/devilan/id544777502?l=de&ls=1&mt=8#
#
# - Use Android App DeviLAN (TCP/IP based Remote Control tool) to find and control
#   Raspberry Pis which you can download for free from Google play
#   https://play.google.com/store/apps/details?id=com.synertronixx.devilan
#
# - Also use program devilan_pi_rc_server_piface.py as example to create your own Remote Control GUI
#   for iPhone, iPad and iPod or your Android device to control Raspberry Pi and
#   peripheral devices e.g. "Pi Face"
#
# - Get further information about DeviLAN app from our website:
#   http://www.synertronixx.de/produkte/devilanapp/DeviLANApp_search_configure.htm
#   http://www.synertronixx.de/produkte/devilanapp/DeviLANApp_remote_control.htm
#
# - Testing:
#   1. Copy this file to "/home/pi/DeviLAN_Pi_Servers/"
#   2. Start Python IDLE
#   3. Open this file
#   4. Run the script (F5)
#
# - Start script on power up (boot) automatically:
#   1. Open file  "/etc/rc.local" in editor
#      e.g.: "sudo nano /etc/rc.local"
#   2. Add line "python /home/pi/DeviLAN_Pi_Servers/devilan_pi_finder.py &
#      and store the file
#   3. Make file executable
#      e.g.: "chmod 755 /home/pi/DeviLAN_Pi_Servers/devilan_pi_finder.py
#
# - History:
#   18.03.2014: Released first beta version (quick & dirty)
#   24.04.2014: New Google play links for Android App 
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
# *****************************************************************************/

import socket
import SocketServer
import sys
import struct
import binascii
import fcntl
from uuid import getnode as get_mac


# Some global settings
APPNAME = "DeviLAN-Pi Finder 1.1: " # Application name for debug output
DEVICE_NAME = "Raspberry Pi + Pi Face Digital" # Device name, max. 32 characters
INTERFACE = "eth0"  # LAN interface
HOST = "0.0.0.0"    # Send Broacast datagram
PORT = 8002         # UDP communication port
TCPIP_PORT = 3333   # TCP communication port

# synertronixx UDP datagram commands
UDP_IDENT_DATAGRAM_LEN = 62
UDP_CONFIGURATION_DATAGRAM_LEN = 58
UDP_IDENTIFY_DATAGRAM_LEN = 10

UDP_COMMAND_IDENTIFY_ALL =   0x0001
UDP_COMMAND_IDENTIFY =       0x0002
UDP_COMMAND_IDENTIFICATION = 0x0003
UDP_COMMAND_CONFIGURATION =   0x0004
UDP_COMMAND_REBOOT =          0x0005

# Debug switch
COMM_DEBUG = 1



#------------------------------------------------
# Receive handler
#------------------------------------------------
class packetHandler(SocketServer.BaseRequestHandler):

 def handle(self):
        global COMM_DEBUG
	#data = self.request[0].strip()
        data = self.request[0]

        if COMM_DEBUG == 1:
            print ("") 
            print (APPNAME + "Received a packet from %s (%d Bytes)" % (str(self.client_address), len(data)) )
            print (APPNAME + "Received data: " + " ".join(hex(ord(n)) for n in data) )

       # extract data from received data
       # commmand, MAC (high, low), command, length

        if (len(data) < 10):
            print (APPNAME + "Invalid datagram length, stop datagram processing !")
            return
        
        # copy first 10 bytes
        datax = data[0:10]
        #extract header information (Command, MAC (High, Low), Length)
        values = struct.Struct(">H H I H")
        unpacked_data = values.unpack(datax)
        rcv_mac = 0x000000000000 + (unpacked_data[1] << 32) +  unpacked_data[2]

        if COMM_DEBUG == 1:
            print (APPNAME + "Unpacked Values: %s" % str(unpacked_data) )
            print (APPNAME + "Command: %04X" % unpacked_data[0])
            print (APPNAME + "MAC:     %012X" % rcv_mac)
            print (APPNAME + "Length:  %4d" % unpacked_data[3])

       # verify MAC and command
        own_mac = get_mac()

        if (rcv_mac != own_mac) & (unpacked_data[0] == UDP_COMMAND_IDENTIFY_ALL) & (unpacked_data[3] == UDP_IDENTIFY_DATAGRAM_LEN):
            print (APPNAME + "MAC is different and command asks for identification" )

            socket = self.request[1]
            try:
                packMsg = create_datagram()
                socket.sendto(packMsg, self.client_address)

                print (APPNAME + "Sending from %s to %s" % (get_ip_address(INTERFACE), self.client_address) )
            except(socket.timeout, socket.error) as e:
                print (APPNAME + "Error when responding - quitting...")
                sys.exit(1)

        else:
            if (own_mac == rcv_mac) & (unpacked_data[0] == UDP_COMMAND_REBOOT):
                print (APPNAME + "Received Reboot datagram (not handled)")
                print (APPNAME + "Insert code to reboot Raspberry Pi here")
            else:
                if (own_mac == rcv_mac) & (unpacked_data[0] == UDP_COMMAND_CONFIGURATION) & (unpacked_data[3] >= UDP_CONFIGURATION_DATAGRAM_LEN):
                    print (APPNAME + "Received Configuration datagram (not handled)")
                    print (APPNAME + "Only module name will be changed")
                    print (APPNAME + "Insert code to change other configurations here")
                    values = struct.Struct(">H H I H B I H I I 32s B")
                    unpacked_data2 = values.unpack(data)
                    global DEVICE_NAME
                    # DEVICE_NAME =  unpacked_data2[9]

                    newname =  unpacked_data2[9].split("\0")
                    DEVICE_NAME = newname[0]
                    print (APPNAME + "New device name \"%s\"" % (DEVICE_NAME) )
                    socket = self.request[1]
                    try:
                        packMsg = create_datagram()
                        socket.sendto(packMsg, self.client_address)

                        print (APPNAME + "Sending %s to %s" % (get_ip_address(INTERFACE), self.client_address) )
                    except(socket.timeout, socket.error) as e:
                        print (APPNAME + "Error when responding - quitting...")
                        sys.exit(1)
                else:
                    print (APPNAME + "Unkown datagram")


#-------------------------------------------------------------
# create_datagram: create 62 bytes UDP identification datagram 
#-------------------------------------------------------------
def create_datagram():

        global COMM_DEBUG
        if COMM_DEBUG == 1:
            print (APPNAME + "Create identification datagram")
  
	# get the MAC, 2 bytes high, 4 bytes low
        high_mac =  get_mac()
        high_mac = high_mac >> 32
        low_mac = get_mac()
        low_mac = low_mac & 0xFFFFFFFF

        if COMM_DEBUG == 1:
            print (APPNAME + "High Mac: %04X" % high_mac)
            print (APPNAME + "Low Mac : %08X" % low_mac)

        # command ID
        cmd_value = UDP_COMMAND_IDENTIFICATION
        # length of identification datagram in bztes
        cmd_len = UDP_IDENT_DATAGRAM_LEN
        own_ip =  iptoint(get_ip_address(INTERFACE) )

        # Device name
        own_name = DEVICE_NAME

        # unused parameter, only to be compatible to DeviLAN App protocol
        # if necessary insert code here to get DHCP, netmask, gateway or serial no. infomation 
        own_dhcp = int("1")
        own_netmask = 0xFFFFFF00
        own_gateway = 0xC0A801FE
        own_serial =  0x00000001

        # '>' = Big Endian
        # Ident [2], MAC (high, low [6]), length (62)[2], DHCP[1], IP[4], Port[2], Netmask[4], Gateway[4], Name, 0x00, Serial
        packMsg = struct.pack(">HHIHBIHII32sBI", cmd_value, high_mac, low_mac, cmd_len,
        own_dhcp, own_ip, TCPIP_PORT, own_netmask, own_gateway, own_name, 0x00, own_serial);

        if COMM_DEBUG == 1:
            print (APPNAME + "Length datagram: %d" % len(packMsg) )
            print (APPNAME + "Packed Value   : %s" % (binascii.hexlify(packMsg)) )

        return  packMsg

#------------------------------------------------
# iptoint: convert dotted IP string (ip) to int
#------------------------------------------------
def iptoint(ip_str):
    return int(socket.inet_aton(ip_str).encode('hex'),16)

#------------------------------------------------
# inttoip: convert int (ip) to dotted IP string
#------------------------------------------------
def inttoip(ip):
    return socket.inet_ntoa(hex(ip)[2:-1].decode('hex'))


#--------------------------------------------------------------
# get_ip_address: get IP address for a given interface (ifname)
#--------------------------------------------------------------
def get_ip_address(ifname):
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack('256s', ifname[:15])
    )[20:24])


#------------------------------------------------
# main
#------------------------------------------------
def main():
	try:
		server = SocketServer.UDPServer((HOST, PORT), packetHandler)

		print (APPNAME + "Starting synertronixx UDP identification server")
		print (APPNAME + "Waiting for incomming UDP datagrams at port " + str(PORT) )

		server.serve_forever()
	except socket.error as e:
		print (APPNAME + "Error, Server not started : %s" % str(e) )
		sys.exit(1)

# main
# own_host = socket.gethostbyname(socket.gethostname())

main()


# End of file

